program CHEBYSHEV;
{--------------------------------------------------------------------}
{  Alg4'6.pas   Pascal program for implementing Algorithm 4.6        }
{                                                                    }
{  NUMERICAL METHODS: Pascal Programs, (c) John H. Mathews 1995      }
{  To accompany the text:                                            }
{  NUMERICAL METHODS for Math., Science & Engineering, 2nd Ed, 1992  }
{  Prentice Hall, Englewood Cliffs, New Jersey, 07632, U.S.A.        }
{  Prentice Hall, Inc.; USA, Canada, Mexico ISBN 0-13-624990-6       }
{  Prentice Hall, International Editions:   ISBN 0-13-625047-5       }
{  This free software is compliments of the author.                  }
{  E-mail address:       in%"mathews@fullerton.edu"                  }
{                                                                    }
{  Algorithm 4.6 (Chebyshev Approximation).                          }
{  Section   4.5, Chebyshev Polynomials, Page 246                    }
{--------------------------------------------------------------------}

  uses
    crt;

  const
    GNmax = 130;
    MaxR = 50;
    FunMaX = 6;

  type
    MATRIX = array[0..MaxR, 0..MaxR] of real;
    VECTOR = array[0..MaxR] of real;
    RVECTOR = array[0..GNmax] of real;
    LETTER = string[8];
    LETTERS = string[200];
    STATUS = (Computing, Done, More, SawTable, Working);

  var
    FunType, GNpts, Inum, J, K, N, Sub: INTEGER;
    Rnum, Z: real;
    C, P, X, Y: VECTOR;
    Xg, Yg, Yf: RVECTOR;
    T: MATRIX;
    Ans: LETTER;
    Mess: LETTERS;
    Stat, State: STATUS;

  function F (X: real): real;
  begin
    case FunType of
      1:
        F := EXP(X);
      2: 
        F := COS(X);
      3: 
        F := SIN(X);
      4: 
        F := SIN(X) / COS(X);
      5: 
        F := ARCTAN(X);
      6: 
        begin
          if X = -1 then
            F := -1E+36
          else
            F := LN(ABS(X + 1));
        end;
    end;
  end;

  procedure PRINTFUN (FunType: integer);
  begin
    case FunType of
      1: 
        WRITE('EXP(X)');
      2: 
        WRITE('COS(X)');
      3: 
        WRITE('SIN(X)');
      4: 
        WRITE('TAN(X)');
      5: 
        WRITE('ARCTAN(X)');
      6: 
        WRITE('LN(X+1)');
    end;
  end;

  procedure CHEBY (var C: VECTOR; N: integer);
    var
      J, K: integer;
      D, Sum, Z: real;
      Y: VECTOR;
  begin
    D := PI / (2 * N + 2);
    for K := 0 to N do
      begin
        X[K] := COS((2 * K + 1) * D);
        Y[K] := F(X[K]);
        C[K] := 0;
      end;
    for K := 0 to N do
      begin
        Z := (2 * K + 1) * D;
        for J := 0 to N do
          C[J] := C[J] + Y[K] * COS(J * Z);
      end;
    C[0] := C[0] / (N + 1);
    for J := 1 to N do
      C[J] := 2 * C[J] / (N + 1);
  end;

  function CHEPOLY (X: real): real;
    var
      J: integer;
      Sum: real;
      T: VECTOR;
  begin
    T[0] := 1;
    T[1] := X;
    if N > 1 then
      for J := 1 to N - 1 do
        T[J + 1] := 2 * X * T[J] - T[J - 1];
    Sum := 0;
    for J := 0 to N do
      Sum := Sum + C[J] * T[J];
    CHEPOLY := Sum;
  end;

  procedure CREATEPOLYS (var T: MATRIX; N: integer);
    var
      J, K: integer;
  begin
    for K := 0 to N do
      for J := 0 to N do
        T[K, J] := 0;
    T[0, 0] := 1;
    T[1, 0] := 0;
    T[1, 1] := 1;
    for K := 2 to N do
      begin
        for J := 1 to K do
          T[K, J] := 2 * T[K - 1, J - 1] - T[K - 2, J];
        T[K, 0] := -T[K - 2, 0];
      end;
  end;

  procedure MAKECHEPOLY (T: MATRIX; C: VECTOR; var P: VECTOR; N: integer);
    var
      J, K: integer;
      Sum: real;
  begin
    for J := 0 to N do
      begin
        Sum := 0;
        for K := 0 to N do
          Sum := Sum + C[K] * T[K, J];
        P[J] := Sum;
      end;
  end;

  procedure MESSAGE;
    var
      Ans: CHAR;
  begin
    CLRSCR;
    WRITELN('                          CHEBYSHEV POLYNOMIALS');
    WRITELN;
    WRITELN;
    WRITELN('          Construction of the Chebyshev series for the function F(x):');
    WRITELN;
    WRITELN('              P(x) = c T (x) + c T (x) + c T (x) +...+ c T (x)');
    WRITELN('                      0 0       1 1       2 2           N N   ');
    WRITELN;
    WRITELN('     over the interval [-1,1].');
    WRITELN;
    WRITELN;
    WRITELN('     The Chebyshev polynomials used in the above formula are:');
    WRITELN;
    WRITELN('                                         2                  3');
    WRITELN('     T (x) = 1 ,  T (x) = x ,  T (x) = 2x  - 1 ,  T (x) = 4x  - 3x');
    WRITELN('      0            1            2                  3');
    WRITELN('               4     2                    5      3');
    WRITELN('     T (x) = 8x  - 8X  + 1  ,  T (x) = 16X  - 20X  + 5X     , ... ,');
    WRITELN('      4                         5 ');
    WRITELN;
    WRITELN('     and are generated using   T (x) = 2 x T   (x) - T   (x)   for j=>2.');
    WRITELN('                                j           j-1       j-2');
    WRITELN;
    WRITELN;
    WRITE('                          Press the <ENTER> key. ');
    READLN(Ans);
    WRITELN;
    CLRSCR;
    WRITELN('     The coefficients { c  } are computed using the formulas:');
    WRITELN('                         j');
    WRITELN;
    WRITELN;
    WRITELN('               N                     N                      ');
    WRITELN('           1                     2                   2k+1   ');
    WRITELN('     c  = --- Sum F(x ) ,  c  = --- Sum F(x )Cos(J*Pi----)  ');
    WRITELN('      0   N+1        k      j   N+1        k         2N+1   ');
    WRITELN('              k=0                   k=0                     ');
    WRITELN;
    WRITELN;
    WRITELN('     for j=1,2,...,N    where  x  = Cos(Pi*(2k+1)/(2N+2))');
    WRITELN('                                k');
    WRITELN;
    WRITELN;
    WRITE('     Press the <ENTER> key. ');
    READLN(Ans);
    WRITELN;
  end;

  procedure GETN (var N: integer; FunType: integer);
  begin
    CLRSCR;
    WRITELN;
    WRITELN;
    WRITELN('     You chose to construct the Chebyshev series for ');
    WRITELN;
    case FunType of
      1: 
        WRITE('     F(X)  =  EXP(X)');
      2: 
        WRITE('     F(X)  =  COS(X)');
      3: 
        WRITE('     F(X)  =  SIN(X)');
      4: 
        WRITE('     F(X)  =  TAN(X)');
      5: 
        WRITE('     F(X)  =  ARCTAN(X)');
      6: 
        WRITE('     F(X)  =  LN(X+1)');
    end;
    WRITELN(' over the interval [-1,1].');
    WRITELN;
    WRITELN;
    WRITELN('     F(x)  ~  c T (x) + c T (x) + c T (x) +...+ c T (x).');
    WRITELN('               0 0       1 1       2 2           N N   ');
    WRITELN;
    WRITELN;
    Mess := '     ENTER the degree  N = ';
    N := 5;
    WRITE(Mess);
    READLN(N);
    WRITELN;
    if N < 1 then
      N := 1;
    if N > MaxR then
      N := MaxR;
    WRITELN;
  end;

  procedure GETFUNCTION (var FunType: integer);
    var
      K: integer;
  begin
    FunType := 0;
    while FunType = 0 do
      begin
        CLRSCR;
        WRITELN;
        WRITELN;
        WRITELN('     The Chebyshev series will be constructed for:');
        WRITELN;
        WRITELN;
        for K := 1 to FunMax do
          begin
            WRITE('     <', K : 2, ' >  F(X) = ');
            PRINTFUN(K);
            WRITELN;
            WRITELN;
          end;
        WRITELN;
        WRITELN;
        Mess := '            SELECT < 1 - 6 > ?  ';
        FunType := 1;
        WRITE(Mess);
        READLN(FunType);
        if FunType < 1 then
          FunType := 1;
        if FunType > 6 then
          FunType := 6;
      end;
  end;

  procedure PRINTPOLY (A: VECTOR; N: integer);
    var
      K, U, V: integer;
  begin
    WRITELN;
    WRITELN('     The Chebyshev polynomial approximation of degree  N = ', N : 2, '  is:');
    WRITELN;
    case N of
      1: 
        begin
          WRITELN('P(x)  =  a  +  a x');
          WRITELN('          0     1');
        end;
      2: 
        begin
          WRITELN('                           2');
          WRITELN('P(x)  =  a   +  a x  +  a x');
          WRITELN('          0      1       2');
        end;
      3: 
        begin
          WRITELN('                           2        3');
          WRITELN('P(x)  =  a   +  a x  +  a x  +  a  x');
          WRITELN('          0      1       2       3');
        end;
      4, 5, 6, 7, 8, 9: 
        begin
          WRITELN('                           2            ', N - 1 : 1, '        ', N : 1);
          WRITELN('P(x)  =  a   +  a x  +  a x   +...+  a x   +  a x');
          WRITELN('          0      1       2            ', N - 1 : 1, '        ', N : 1);
        end;
      10: 
        begin
          WRITELN('                           2            ', N - 1 : 1, '         ', N : 2);
          WRITELN('P(x)  =  a   +  a x  +  a x   +...+  a x   +  a  x');
          WRITELN('          0      1       2            ', N - 1 : 1, '        ', N : 2);
        end;
      else
        begin
          WRITELN('                           2             ', N - 1 : 2, '        ', N : 2);
          WRITELN('P(x)  =  a   +  a x  +  a x   +...+  a  x   +  a  x');
          WRITELN('          0      1       2            ', N - 1 : 2, '        ', N : 2);
        end;
    end;
    WRITELN;
    for K := 0 to TRUNC(N / 2) do                {Print the coefficients}
      begin
        U := 2 * K;                                {of P(X) in two columns}
        V := 2 * K + 1;
        if U <= N then
          begin
            WRITE('A(', U : 2, ' ) =', A[U] : 15 : 7, '         ');
            if V <= N then
              WRITELN('A(', V : 2, ' ) =', A[V] : 15 : 7)
            else
              WRITELN;
          end;
      end;
  end;

  procedure PRINTCHEB (A: VECTOR; N: integer);
    var
      K, U, V: integer;
  begin
    WRITELN;
    WRITELN('     The Chebyshev series with N =', N : 3, ' terms is:');
    WRITELN;
    WRITELN;
    case N of
      1: 
        begin
          WRITELN('P(x)  =  c  +  c T (x)');
          WRITELN('          0     1 1');
        end;
      2: 
        begin
          WRITELN('P(x)  =  c   +  c T (x)  +  c T (x)');
          WRITELN('          0      1 1         2 2');
        end;
      3: 
        begin
          WRITELN('P(x)  =  c   +  c T (x)  +  c T (x)  +  c T (x)');
          WRITELN('          0      1 1         2 2         3 3');
        end;
      4, 5, 6, 7, 8, 9: 
        begin
          WRITELN('P(x)  =  c  + c T (x) + c T (x)  +...+  c T (x) + c T (x)');
          WRITELN('          0    1 1       2 2             ', N - 1 : 1, ' ', N - 1 : 1, '       ', N : 1, ' ', N : 1);
        end;
      10: 
        begin
          WRITELN('P(x)  =  c  + c T (x) + c T (x)  +...+  c T (x)  + c  T  (x)');
          WRITELN('          0    1 1       2 2             ', N - 1 : 1, ' ', N - 1 : 1, '        ', N : 2, ' ', N : 2);
        end;
      else
        begin
          WRITELN('P(x)  =  c  + c T (x) + c T (x)  +...+  c  T  (x)  + c  T  (x)');
          WRITELN('          0    1 1       2 2             ', N - 1 : 2, ' ', N - 1 : 2, '        ', N : 2, ' ', N : 2);
        end;
    end;
    WRITELN;
    for K := 0 to TRUNC(N / 2) do                {Print the coefficients}
      begin
        U := 2 * K;                                {of P(X) in two columns}
        V := 2 * K + 1;
        if U <= N then
          begin
            WRITE('C(', U : 2, ' ) =', A[U] : 15 : 7, '         ');
            if V <= N then
              WRITELN('C(', V : 2, ' ) =', A[V] : 15 : 7)
            else
              WRITELN;
          end;
      end;
  end;

  procedure RESULT (T: MATRIX; A, C: VECTOR; N, FunType: integer);
  begin
    CLRSCR;
    WRITELN;
    WRITELN;
    WRITELN;
    case FunType of
      1: 
        WRITE('     F(X) = EXP(X)');
      2: 
        WRITE('     F(X) = COS(X)');
      3: 
        WRITE('     F(X) = SIN(X)');
      4: 
        WRITE('     F(X) = TAN(X)');
      5: 
        WRITE('     F(X) = ARCTAN(X)');
      6: 
        WRITE('     F(X) = LN(X+1)');
    end;
    WRITELN(' over the interval [-1,1].');
    WRITELN;
  end;

  procedure TABLE (T: MATRIX; A, C: VECTOR; N, FunType: integer; var State: STATUS);
    var
      J, ML, MN: integer;
      H, Z: real;
  begin
    CLRSCR;
    WRITELN;
    WRITELN;
    if State = Computing then
      begin
        WRITELN;
        WRITE('Want  to  construct   a  table  of  approximations ?  <Y/N>  ');
        READLN(Ans);
      end;
    if State = More then
      Ans := 'Y';
    if (Ans = 'Y') or (Ans = 'y') then
      begin
        State := SawTable;
        WRITELN;
        WRITELN('A table of M+1 equally spaced point will be made in [-1,1].');
        Mess := 'ENTER the number of subintervals  M = ';
        MN := 10;
        WRITE(Mess);
        READLN(MN);
        if MN < 1 then
          MN := 1;
        if MN > 100 then
          MN := 100;
        WRITELN;
        H := 2 / MN;
        CLRSCR;
        WRITELN;
        if N < 10 then
          begin
            WRITELN('         x                F(x )            P (x )');
            WRITELN('          k                  k              ', N : 1, '  k');
          end
        else
          begin
            WRITELN('         x                F(x )            P  (x )');
            WRITELN('          k                  k              ', N : 2, '  k');
          end;
        WRITELN('     ------------------------------------------------');
        ML := 0;
        if FunType = 6 then
          ML := 1;
        for J := ML to MN do
          begin
            if MN < 8 then
              WRITELN;
            Z := -1 + H * J;
            if K = MN then
              Z := 1;
            WRITELN(Z : 17 : 8, F(Z) : 17 : 8, CHEPOLY(Z) : 17 : 8);
          end;
      end;
  end;

begin                                            {Begin Main Program}
  MESSAGE;
  Stat := Working;
  while (Stat = Working) do
    begin
      GETFUNCTION(FunType);
      GETN(N, FunType);
      CHEBY(C, N);
      CREATEPOLYS(T, N);
      MAKECHEPOLY(T, C, P, N);
      RESULT(T, P, C, N, FunType);
      PRINTCHEB(C, N);
      WRITELN;
      WRITE('Press the <ENTER> key.  ');
      READLN(Ans);
      WRITELN;
      CLRSCR;
      RESULT(T, P, C, N, FunType);
      PRINTPOLY(P, N);
      WRITELN;
      WRITE('Press the <ENTER> key.  ');
      READLN(Ans);
      State := Computing;
      while (State = Computing) or (State = More) or (State = Sawtable) do
        begin
          CLRSCR;
          TABLE(T, P, C, N, FunType, State);
          if State = SawTable then
            begin
              WRITELN;
              WRITE('Press the <ENTER> key.  ');
              READLN(Ans);
              WRITELN;
              WRITELN;
              WRITE('Do you want to see another table of approximations ?  <Y/N>  ');
              Ans := 'N';
              READLN(Ans);
              WRITELN;
            end
          else
            begin
              Ans := 'N';
              WRITELN;
            end;
          if (Ans <> 'Y') and (Ans <> 'y') then
            State := Done;
          if (Ans = 'Y') or (Ans = 'y') then
            State := More;
        end;
      WRITELN;
      WRITELN;
      WRITE('Do you want to find a different  Chebyshev  series ?  <Y/N>  ');
      Ans := 'N';
      READLN(Ans);
      if (Ans <> 'Y') and (Ans <> 'y') then
        Stat := Done;
    end;
end.                                               {End Main Program}

